(************************************************************************
 *                                                                      *
 *   Ant Movie Catalog 4.x                                              *
 *   (C) 2000-2011 Antoine Potten, Mickal Vanneufville                 *
 *   http://www.antp.be/software                                        *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 *   This program is free software; you can redistribute it and/or      *
 *   modify it under the terms of the GNU General Public License        *
 *   as published by the Free Software Foundation; either version 2     *
 *   of the License, or (at your option) any later version.             *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 ************************************************************************)

unit framemoviecustom;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls, Contnrs,

  AntJvEdit, AntJvSpin, AntJvExControls, AntJvToolEdit,

  movieclass, ConstValues, ComCtrls, ActnList, TB2Item, TBX, Menus;

const
  CF_FrameMarginLeft = 6;
  CF_FrameMarginRight = 1;
  CF_FrameMarginTop = 3;
  CF_PanelMarginLeft =  2;
  CF_PanelMarginRight = 2;
  CF_PanelMarginTop = 2;
  CF_PanelMarginBottom = 2;
  CF_PanelBorderSize = 2;
  CF_PanelHeight = 25;
  CF_PanelHeightText = 55;
  CF_PanelWidth = 200;
  CF_PanelMinWidth = 32;
  CF_PanelMaxWidth = 0; //infinite
  CF_PanelMinHeight = 25;
  CF_PanelMaxHeight = 0; //infinite
  CF_LabelWidth = 94;
  CF_LabelMinWidth = 12;
  CF_LabelMaxWidth = 0; //infinite
  CF_PanelMarginBetween = 4; //margin between label and control

var
  CF_FrameWidthRef: Integer = 533;
  CF_FrameHeightRef: Integer = 449;

type
  TPanelCustomField = class(TPanel)
    procedure FieldChange(Sender: TObject);
    procedure FieldKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FieldKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FielURLButtonClick(Sender: TObject);

  private
    FEnabled: Boolean;
    FBorder: Boolean;
    FType: TFieldType;
    FName: String;
    FLabel: TLabel;
    FControl: TControl;

    FOnFieldChange: TNotifyEvent;
    FOnUserAskListUpdate: TNotifyEvent;
    FOnURLButtonClick: TNotifyEvent;
    FOnURLEnter: TNotifyEvent;

    procedure FSetEnabled(Value: Boolean);
    procedure FSetBorder(Value: Boolean);
    procedure FSetType(Value: TFieldType);
    procedure FSetName(Value: String);
    function FGetValue() : String;
    procedure FSetValue(Value: String);
    function FGetLabelWidth() : Integer;
    procedure FSetLabelWidth(Value: Integer);
  public
    Properties: TCustomFieldProperties; // ref on FieldProperties in movieclass

    // Custom Field GUIProperties
    AutoStretchWidth: Boolean;
    AutoStretchHeight: Boolean;
    PosXRef: Integer; // according to const value CF_FrameWidthRef
    PosYRef: Integer; // according to const value CF_FrameHeightRef
    WidthRef: Integer; // according to const value CF_FrameWidthRef
    HeightRef: Integer; // according to const value CF_FrameHeightRef

    // Calculated values for frame resize
    NbAutoStretchOnLine: Integer;
    NbAutoStretchOnLeft: Integer;
    NbAutoStretchOnColumn: Integer;
    NbAutoStretchOnTop: Integer;

    constructor Create(AOwner: TComponent); override;
    procedure Init(FieldProperties: TCustomFieldProperties);
    procedure LoadGUIProperties;
    procedure SaveGUIProperties;
    procedure PanelResize(Sender: TObject);
    property CFEnabled: Boolean read FEnabled write FSetEnabled;
    property CFBorder: Boolean read FBorder write FSetBorder;
    property CFType: TFieldType read FType write FSetType;
    property CFName: string read FName write FSetName;
    property CFValue: string read FGetValue write FSetValue;
    property CFLabelWidth: Integer read FGetLabelWidth write FSetLabelWidth;

    property OnFieldChange: TNotifyEvent read FOnFieldChange write FOnFieldChange;
    property OnUserAskListUpdate: TNotifyEvent read FOnUserAskListUpdate write FOnUserAskListUpdate;
    property OnURLButtonClick: TNotifyEvent read FOnURLButtonClick write FOnURLButtonClick;
    property OnURLEnter: TNotifyEvent read FOnURLEnter write FOnURLEnter;
  end;

  TMovieFrameCustom = class(TFrame)
    ActionList1: TActionList;
    ActionAddCustomField: TAction;
    ActionDeleteCustomField: TAction;
    ActionMoveResizeCustomFields: TAction;
    ActionModifyCustomField: TAction;
    PopupCustomField: TTBPopupMenu;
    TBModifyCustomField: TTBXItem;
    TBDeleteCustomField: TTBXItem;
    ActionDefaultPositioning: TAction;
    ActionAutoStretchWidth: TAction;
    ActionAutoStretchHeight: TAction;
    TBSep1: TTBXSeparatorItem;
    TBAutoStretchWidth: TTBXItem;
    TBAutoStretchHeight: TTBXItem;
    TBAddCustomField: TTBXItem;
    TBMoveResizeCustomField: TTBXItem;
    TBSep2: TTBXSeparatorItem;
    TBDefaultPositioning: TTBXItem;

    procedure FieldChange(Sender: TObject);
    procedure FieldPropertiesChange(Sender: TObject);
    procedure UserAskListUpdate(Sender: TObject);
    procedure FielURLEnter(Sender: TObject);
    procedure FielURLButtonClick(Sender: TObject);

    procedure FrameResize(Sender: TObject);

    procedure ActionMoveResizeCustomFieldsExecute(Sender: TObject);
    procedure ActionAddCustomFieldExecute(Sender: TObject);
    procedure ActionDeleteCustomFieldExecute(Sender: TObject);

    procedure ControlMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ControlMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ControlMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure FrameMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ActionModifyCustomFieldExecute(Sender: TObject);
    procedure ActionDefaultPositioningExecute(Sender: TObject);
    procedure ActionAutoStretchWidthExecute(Sender: TObject);
    procedure ActionAutoStretchHeightExecute(Sender: TObject);
  private
    FModified: Boolean;
    FPropertiesModified: Boolean;
    FUpdateCount: Integer;
    OldPos: TPoint;
    APanelCF: TPanelCustomField;
    Resizing: Integer;
    FAllowEdit: Boolean;
    FSelectedPanel: TPanelCustomField;
    FGeneratedFields: Boolean;

    FOnFieldChange: TNotifyEvent;
    FOnFieldPropertiesChange: TNotifyEvent;
    FOnFieldAdd: TNotifyEvent;
    FOnFieldDelete: TNotifyEvent;
    FOnFieldModify: TNotifyEvent;
    FOnUserAskListUpdate: TNotifyEvent;
    FOnURLButtonClick: TNotifyEvent;
    FOnURLEnter: TNotifyEvent;

    procedure SetAllowEdit(Value: Boolean);
    procedure SetFieldsEnabled(Value: Boolean);
    procedure SetFieldsBorder(Value: Boolean);

    procedure CheckFieldsPosition;
    procedure ComputeFieldsValue;
    procedure UpdateFieldsProperties;
    function AddField(FieldProperties: TCustomFieldProperties): TPanelCustomField; overload;

  public
    Properties: TCustomFieldsProperties; //ref on FieldsProperties in movieclass

    constructor Create(AOwner: TComponent); override;
    procedure SetImageList(AImageList: TImageList);

    procedure ClearFields;
    procedure GenerateFields;
    procedure LoadFieldsProperties;
    procedure SaveFieldsProperties;

    procedure SetDefaultValues;
    procedure LoadFromObject(AMovie: TMovie);
    procedure SaveToObject(AMovie: TMovie; const ForceSave: Boolean = False);

    function GetCurrentValue(FieldTag: string): string;
    procedure SetCurrentValue(FieldTag: string; FieldValue: string);
    function FindURLControl(str: string): TAntJvComboEditXP;

    property Modified: Boolean read FModified write FModified;
    property PropertiesModified: Boolean read FPropertiesModified write FPropertiesModified;

    procedure BeginUpdate;
    procedure EndUpdate;
    function IsUpdating: Boolean;

    property AllowEdit: Boolean read FAllowEdit write SetAllowEdit;
    procedure Translate;

    property OnFieldChange: TNotifyEvent read FOnFieldChange write FOnFieldChange;
    property OnFieldPropertiesChange: TNotifyEvent read FOnFieldPropertiesChange write FOnFieldPropertiesChange;
    property OnFieldAdd: TNotifyEvent read FOnFieldAdd write FOnFieldAdd;
    property OnFieldDelete: TNotifyEvent read FOnFieldDelete write FOnFieldDelete;
    property OnFieldModify: TNotifyEvent read FOnFieldModify write FOnFieldModify;
    property OnUserAskListUpdate: TNotifyEvent read FOnUserAskListUpdate write FOnUserAskListUpdate;
    property OnURLButtonClick: TNotifyEvent read FOnURLButtonClick write FOnURLButtonClick;
    property OnURLEnter: TNotifyEvent read FOnURLEnter write FOnURLEnter;
  end;

implementation

uses
  Global, StrUtils, functions_str, functions_files;

{$R *.dfm}

{-------------------------------------------------------------------------------
  TPanelCustomField
-------------------------------------------------------------------------------}

constructor TPanelCustomField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Top := CF_FrameMarginTop;
  Left := CF_FrameMarginLeft;
  Height := CF_PanelHeight;
  Width := CF_PanelWidth;
  Constraints.MinHeight := Height;
  Constraints.MaxHeight := Height;
  Constraints.MinWidth := CF_PanelMinWidth;
  Constraints.MaxWidth := CF_PanelMaxWidth;
  ParentColor := True;
  ParentBackground := True;
  BevelOuter := bvNone;
  //Ctl3D := False;
  //ParentCtl3D := False;
  OnResize := PanelResize;

  FLabel := nil;
  FControl := nil;

  FName := '';
  FType := ftString;
  Properties := nil;

  FSetEnabled(False);
  FSetBorder(False);

  PosXRef := -1;
  PosYRef := -1;
  WidthRef := -1;
  HeightRef := -1;
  AutoStretchWidth := True;
  AutoStretchHeight := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.Init(FieldProperties: TCustomFieldProperties);
begin
  Properties := FieldProperties;
  if(Properties = nil) then
    Exit;
  Name := 'P' + Properties.FieldTag;
  Caption := '';
  FSetType(FieldProperties.FieldType);
  FSetName(FieldProperties.FieldName);
  FSetValue('');
  FieldProperties.FieldObject := Self; //ref
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.SaveGUIProperties;
begin
  if(Properties = nil) then
    Exit;
  Properties.GUIProperties := 'rx' + FloatToStr(PosXRef, FormatSettings) +
    ':ry' + FloatToStr(PosYRef, FormatSettings) +
    ':rw' + FloatToStr(WidthRef, FormatSettings) +
    ':rh' + FloatToStr(HeightRef, FormatSettings) +
    ':aw' + IfThen(AutoStretchWidth, '1', '0') +
    ':ah' + IfThen(AutoStretchHeight, '1', '0') +
    ':lw' + IntToStr(FGetLabelWidth);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.LoadGUIProperties;
var
  s, tag, value: string;
  p, LabelWidth: Integer;
begin
  if(Properties = nil) then
    Exit;
  PosYRef := -1;
  PosXRef := -1;
  WidthRef := -1;
  HeightRef := -1;
  AutoStretchWidth := True;
  AutoStretchHeight := False;
  LabelWidth := -1;
  s := Properties.GUIProperties;
  while s <> '' do
  begin
    tag := copy(s, 1, 2);
    p := pos(':', s);
    if p = 0 then
    begin
      p := Length(s);
      value := copy(s, 3, p - 2);
    end else
      value := copy(s, 3, p - 3);
    delete(s, 1, p);
    if tag = 'ry' then PosYRef := StrToIntDef(value, -1)
    else if tag = 'rx' then PosXRef := StrToIntDef(value, -1)
    else if tag ='rw' then WidthRef := StrToIntDef(value, -1)
    else if tag = 'rh' then HeightRef := StrToIntDef(value, -1)
    else if tag = 'aw' then AutoStretchWidth := (value = '1')
    else if tag = 'ah' then AutoStretchHeight := ((value = '1') and (FType = ftText))
    else if tag = 'lw' then LabelWidth := StrToIntDef(value, -1);
  end;
  if PosXRef < -1 then PosXRef := -1;
  if PosYRef < -1 then PosYRef := -1;
  if WidthRef = -1 then WidthRef := CF_PanelWidth
  else if WidthRef < CF_PanelMinWidth then WidthRef := CF_PanelMinWidth;
  if (FType <> ftText) then HeightRef := CF_PanelHeight
  else if HeightRef = -1 then HeightRef := CF_PanelHeightText
  else if HeightRef < CF_PanelMinHeight then HeightRef := CF_PanelMinHeight;
  if LabelWidth = -1 then LabelWidth := CF_LabelWidth
  else if LabelWidth < CF_LabelMinWidth then LabelWidth := CF_LabelMinWidth;
  FSetLabelWidth(LabelWidth);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetEnabled(Value: Boolean);
begin
  FEnabled := Value;
  if(FLabel <> nil) then
    FLabel.Enabled := FEnabled;
  if(FControl <> nil) then
    FControl.Enabled := FEnabled;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetBorder(Value: Boolean);
begin
  FBorder := Value;
  if FBorder then
    BorderStyle := bsSingle
  else
    BorderStyle := bsNone;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetType(Value: TFieldType);
var
  i: Integer;
begin
  if (Value <> FType) or (FControl = nil) or (FLabel = nil) then
  begin
    if FControl <> nil then
    begin
      FControl.Hide;
      FControl.Free;
    end;
    if FLabel = nil then
    begin
      FLabel := TLabel.Create(Self);
      FLabel.Parent := Self;
      if Properties <> nil then
        FLabel.Name := 'L'+Properties.FieldTag;
      FLabel.Visible := True;
      FLabel.Caption := '';
      FLabel.Width := CF_LabelWidth;
      FLabel.Constraints.MinWidth := CF_LabelMinWidth;
      FLabel.Constraints.MaxWidth := CF_LabelMaxWidth;
      FLabel.Layout := tlCenter;
      FLabel.AutoSize := False;
      FSetName(FName);
    end;
    Height := CF_PanelHeight;
    Constraints.MinHeight := Height;
    Constraints.MaxHeight := Height;
    case Value of
      ftInteger: //TAntJvSpinEdit
      begin
        FControl := TAntJvSpinEdit.Create(Self);
        FControl.Parent := Self;
        with TAntJvSpinEdit(FControl) do
        begin
          Ctl3D := True;
          MinValue := 0;
          MaxValue := 99999;
          AllowEmpty := True;
          ValueType := AntJvSpin.vtInteger;
          Text := '';
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
        end;
      end;
      ftReal: //TAntJvSpinEdit
      begin
        FControl := TAntJvSpinEdit.Create(Self);
        FControl.Parent := Self;
        with TAntJvSpinEdit(FControl) do
        begin
          Ctl3D := True;
          MinValue := 0;
          MaxValue := 99999;
          AllowEmpty := True;
          ValueType := AntJvSpin.vtFloat;
          Text := '';
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
        end;
      end;
      ftBoolean: //TCheckBox
      begin
        FControl := TCheckBox.Create(Self);
        FControl.Parent := Self;
        with TCheckBox(FControl) do
        begin
          Ctl3D := True;
          Checked := False;
          OnClick := FieldChange;
          OnKeyDown := FieldKeyDown;
          Width := 15;
          //OnKeyUp := FieldKeyUp;
        end;
      end;
      ftDate: //TDateTimePicker
      begin
        FControl := TDateTimePicker.Create(Self);
        FControl.Parent := Self;
        with TDateTimePicker(FControl) do
        begin
          Ctl3D := True;
          ShowCheckbox := True;
          DateTime := SysUtils.Date;
          Checked := False;
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          //OnKeyUp := FieldKeyUp;
        end;
      end;
      ftList: //TComboBox
      begin
        FControl := TComboBox.Create(Self);
        FControl.Parent := Self;
        with TComboBox(FControl) do
        begin
          Ctl3D := True;
          Text := '';
          if (Properties <> nil) then
            for i:=0 to Properties.ListValues.Count-1 do
              Items.Add(Properties.ListValues.Strings[i]);
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
        end;
      end;
      ftText: //TMemo
      begin
        Height := CF_PanelHeight + 30;
        Constraints.MinHeight := CF_PanelMinHeight;
        Constraints.MaxHeight := CF_PanelMaxHeight;
        FControl := TMemo.Create(Self);
        FControl.Parent := Self;
        with TMemo(FControl) do
        begin
          Ctl3D := True;
          Text := '';
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
          ScrollBars := ssVertical;
        end;
      end;
      ftUrl: //TAntJvComboEditXP
      begin
        FControl := TAntJvComboEditXP.Create(Self);
        FControl.Parent := Self;
        with TAntJvComboEditXP(FControl) do
        begin
          ImageKind := ikDropDown;
          Ctl3D := True;
          Text := '';
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
          OnButtonClick := FielURLButtonClick;
        end;
      end;
      else //TEdit
      begin
        FControl := TEdit.Create(Self);
        FControl.Parent := Self;
        with TEdit(FControl) do
        begin
          Ctl3D := True;
          Text := '';
          OnChange := FieldChange;
          OnKeyDown := FieldKeyDown;
          OnKeyUp := FieldKeyUp;
        end;
      end;
    end;
    FType := Value;
    if Properties <> nil then
    begin
      FControl.Name := 'E'+Properties.FieldTag;
      if FControl is TCheckBox then
        TCheckBox(FControl).Caption := '';
    end;
    FControl.Visible := False;
    FSetEnabled(FEnabled);
    PanelResize(Self);
    FControl.Visible := True;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetName(Value: String);
begin
  FName := Value;
  if FLabel <> nil then
  begin
    if(Settings.rOptions.rLanguage.Language = 'French') then
      FLabel.Caption := FName + ' :'
    else
      FLabel.Caption := FName + ':';
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TPanelCustomField.FGetValue: String;
begin
  Result := '';
  if FControl <> nil then
  begin
    try
      case FType of
      ftInteger, ftReal:
        Result := CharReplace(TAntJvSpinEdit(FControl).Text, DecimalSeparator, FormatSettings.DecimalSeparator);
      ftBoolean:
        if TCheckBox(FControl).Checked then
          Result := 'True'
        else
          Result := '';
      ftDate:
        with TDateTimePicker(FControl) do
          if Checked then
            Result := DateToStr(DateTime, FormatSettings);
      ftList:
        Result := TComboBox(FControl).Text;
      ftText:
        Result := TMemo(FControl).Text;
      ftUrl:
        Result := TAntJvComboEditXP(FControl).Text;
      else
        Result := TEdit(FControl).Text;
      end;
    except
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetValue(Value: String);
begin
  if FControl <> nil then
  begin
    try
      case FType of
      ftInteger, ftReal:
        TAntJvSpinEdit(FControl).Text := CharReplace(Value, FormatSettings.DecimalSeparator, DecimalSeparator);
      ftBoolean:
        TCheckBox(FControl).Checked := (Value = 'True');
      ftDate:
        with TDateTimePicker(FControl) do
          if Value = '' then
          begin
            DateTime := SysUtils.Date;
            Checked := False;
          end
          else
          begin
            DateTime := StrToDate(Value, FormatSettings);
            Checked := True;
          end;
      ftList:
        TComboBox(FControl).Text := Value;
      ftText:
        TMemo(FControl).Text := Value;
      ftUrl:
        TAntJvComboEditXP(FControl).Text := Value;
      else
        TEdit(FControl).Text := Value;
      end;
    except
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TPanelCustomField.FGetLabelWidth: Integer;
begin
  Result := 0;
  if FLabel <> nil then
    Result := FLabel.Width;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FSetLabelWidth(Value: Integer);
begin
  if FLabel <> nil then
  begin
    FLabel.Width := Value;
    PanelResize(Self);
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.PanelResize(Sender: TObject);
var
  n: Integer;
begin
  if (FLabel <> nil) and (FControl <> nil) then
  begin
    if FBorder then n := CF_PanelBorderSize else n := 0;
    FLabel.Top := CF_PanelMarginTop - n;
    FLabel.Left := CF_PanelMarginLeft - n;
    FLabel.Height := CF_PanelHeight - CF_PanelMarginTop - CF_PanelMarginBottom;
    FControl.Left :=  FLabel.Width + CF_PanelMarginBetween - n;
    if FType <> ftBoolean then
      FControl.Width :=  Width - FLabel.Width - CF_PanelMarginBetween - CF_PanelMarginRight;
    FControl.Top := CF_PanelMarginTop - n;
    FControl.Height :=  Height - CF_PanelMarginTop - CF_PanelMarginBottom;
    if (FControl is TComboBox) then
      TComboBox(FControl).SelLength := 0;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FieldChange(Sender: TObject);
begin
  if Assigned(FOnFieldChange) then
      FOnFieldChange(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FieldKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = VK_ESCAPE) or ((Key = VK_RETURN) and (FType <> ftText) and (FType <> ftURL)) then
    if Assigned(FOnUserAskListUpdate) then
      FOnUserAskListUpdate(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FieldKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = Ord('A')) and (Shift = [ssCtrl]) then
  begin
    if Sender is TAntJvSpinEdit then
      TAntJvSpinEdit(Sender).SelectAll
    else if Sender is TEdit then
      TEdit(Sender).SelectAll
    else if Sender is TMemo then
      TMemo(Sender).SelectAll
    else if Sender is TComboBox then
      TComboBox(Sender).SelectAll
    else if Sender is TAntJvComboEditXP then
      TAntJvComboEditXP(Sender).SelectAll;
  end
  else if (Key = VK_RETURN) and (Sender is TAntJvComboEditXP) and Assigned(OnURLEnter) then
    OnURLEnter(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TPanelCustomField.FielURLButtonClick(Sender: TObject);
begin
  if Assigned(OnURLButtonClick) then
    OnURLButtonClick(Sender);
end;

{-------------------------------------------------------------------------------
  TMovieFrameCustom
-------------------------------------------------------------------------------}

constructor TMovieFrameCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FModified := False;
  FPropertiesModified := False;
  FGeneratedFields := False;
  FAllowEdit := True;//(not MoveCustomField.Checked);
  FSelectedPanel := nil;
  APanelCF := nil;
  Resizing := 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetImageList(AImageList: TImageList);
begin
  TBAddCustomField.Images := AImageList;
  TBAddCustomField.ImageIndex := Ord(ICON_FIELDADD);
  TBDeleteCustomField.Images := AImageList;
  TBDeleteCustomField.ImageIndex := Ord(ICON_FIELDDEL);
  TBModifyCustomField.Images := AImageList;
  TBModifyCustomField.ImageIndex := Ord(ICON_FIELDEDIT);
  TBMoveResizeCustomField.Images := AImageList;
  TBMoveResizeCustomField.ImageIndex := Ord(ICON_FIELDMOVE);
  TBDefaultPositioning.Images := AImageList;
  TBDefaultPositioning.ImageIndex := Ord(ICON_FIELD);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ClearFields;
var
  i: Integer;
begin
  for i := ControlCount-1 downto 0 do
    if Controls[i] is TPanelCustomField then
      Controls[i].Free;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.GenerateFields;
var
  i: Integer;
  Vis: Boolean;
begin
  FGeneratedFields := False;
  Vis := Visible;
  Visible := False;
  ClearFields;
  if Properties = nil then
    Exit;
  for i := 0 to Properties.Count-1 do
    AddField(Properties.Objects[i]);
  LoadFieldsProperties;
  Visible := Vis;
  FGeneratedFields := True;
  FrameResize(Self);
  FSelectedPanel := nil;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SaveFieldsProperties;
var
  i: Integer;
begin
  if(Properties = nil) then
    Exit;
  Properties.GUIProperties := 'fw' + IntToStr(CF_FrameWidthRef) + ':fh' + IntToStr(CF_FrameHeightRef);
  for i := 0 to ControlCount-1 do
    if Controls[i] is TPanelCustomField then
    begin
      TPanelCustomField(Controls[i]).SaveGUIProperties;
    end;
  FPropertiesModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.LoadFieldsProperties;
var
  s, tag, value: string;
  p, i: Integer;
begin
  if(Properties = nil) then
    Exit;
  CF_FrameWidthRef := -1;
  CF_FrameHeightRef := -1;
  s := Properties.GUIProperties;
  while s <> '' do
  begin
    tag := copy(s, 1, 2);
    p := pos(':', s);
    if p = 0 then
    begin
      p := Length(s);
      value := copy(s, 3, p - 2);
    end else
      value := copy(s, 3, p - 3);
    delete(s, 1, p);
    if tag ='fw' then CF_FrameWidthRef := StrToIntDef(value, -1)
    else if tag = 'fh' then CF_FrameHeightRef := StrToIntDef(value, -1);
  end;
  if CF_FrameWidthRef < 0 then CF_FrameWidthRef := 533;
  if CF_FrameHeightRef < 0 then CF_FrameHeightRef := 449;
  for i := 0 to ControlCount-1 do
    if Controls[i] is TPanelCustomField then
      TPanelCustomField(Controls[i]).LoadGUIProperties;
  CheckFieldsPosition;
  ComputeFieldsValue;
  FPropertiesModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.CheckFieldsPosition;
var
  i: Integer;
  MaxPosYRef: Integer;
  CurrentPanel: TPanelCustomField;
begin
  MaxPosYRef := CF_FrameMarginTop;
  for i := 0 to ControlCount-1 do
    if Controls[i] is TPanelCustomField then
    begin
      CurrentPanel := TPanelCustomField(Controls[i]);
      if (CurrentPanel.PosYRef > 0) and (CurrentPanel.PosYRef + CurrentPanel.HeightRef - 1 > MaxPosYRef) then
        MaxPosYRef := CurrentPanel.PosYRef + CurrentPanel.HeightRef - 1;
    end;

  for i := 0 to Properties.Count-1 do
  begin
    CurrentPanel := TPanelCustomField(Properties.Objects[i].FieldObject);
    if CurrentPanel.PosXRef < 0 then
    begin
      CurrentPanel.PosXRef := CF_FrameMarginLeft;
      if CurrentPanel.AutoStretchWidth then
         CurrentPanel.WidthRef := CF_FrameWidthRef - CF_FrameMarginRight - CurrentPanel.PosXRef;
    end;
    if CurrentPanel.PosYRef < 0 then
    begin
      CurrentPanel.PosYRef := MaxPosYRef;
      MaxPosYRef := CurrentPanel.PosYRef + CurrentPanel.HeightRef - 1;
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.UpdateFieldsProperties;
var
  i: Integer;
  CurrentPanel: TPanelCustomField;
begin
  CF_FrameWidthRef := Width;
  CF_FrameHeightRef := Height;
  for i := 0 to ControlCount-1 do
  if Controls[i] is TPanelCustomField then
  begin
    CurrentPanel := TPanelCustomField(Controls[i]);
    CurrentPanel.PosXRef := CurrentPanel.Left;
    CurrentPanel.PosYRef := CurrentPanel.Top;
    CurrentPanel.WidthRef := CurrentPanel.Width;
    CurrentPanel.HeightRef := CurrentPanel.Height;
  end;
  ComputeFieldsValue;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

// Does not take into account all cases but I think it is enough for normal use
procedure TMovieFrameCustom.ComputeFieldsValue;
var
  i,j,k: Integer;
  CurrentPanel, APanel: TPanelCustomField;
  Panels: TObjectList;
  Found: Boolean;
begin
  Panels := TObjectList.Create(False);
  for i := 0 to ControlCount-1 do
  if Controls[i] is TPanelCustomField then
  begin
    CurrentPanel := TPanelCustomField(Controls[i]);
    CurrentPanel.NbAutoStretchOnLine := 0;
    CurrentPanel.NbAutoStretchOnLeft := 0;
    CurrentPanel.NbAutoStretchOnColumn := 0;
    CurrentPanel.NbAutoStretchOnTop := 0;

    Panels.Clear;
    if CurrentPanel.AutoStretchWidth then
    begin
      CurrentPanel.NbAutoStretchOnLine := 1;
      Panels.Add(CurrentPanel);
    end;
    for j := 0 to ControlCount-1 do
    if (Controls[j] is TPanelCustomField) and (Controls[j] <> Controls[i]) then
    begin
      APanel := TPanelCustomField(Controls[j]);
      if APanel.AutoStretchWidth and not (
        (CurrentPanel.PosYRef >= APanel.PosYRef + APanel.HeightRef - 1) or // CurrentPanel after APanel
        (APanel.PosYRef >= CurrentPanel.PosYRef + CurrentPanel.HeightRef - 1) // CurrentPanel before APanel
        ) then
      begin
        Found := False;
        for k := 0 to Panels.Count-1 do
          with TPanelCustomField(Panels.Items[k]) do
            if not ((APanel.PosXRef >= PosXRef + WidthRef - 1) or
              (PosXRef >= APanel.PosXRef + APanel.WidthRef - 1)) then
              begin
                Found := True;
                Break;
              end;
        if not Found then
        begin
          Panels.Add(APanel);
          Inc(CurrentPanel.NbAutoStretchOnLine);
          if (CurrentPanel.PosXRef >= APanel.PosXRef) then
            Inc(CurrentPanel.NbAutoStretchOnLeft);
        end;
      end;
    end;

    Panels.Clear;
    if CurrentPanel.AutoStretchHeight then
    begin
      CurrentPanel.NbAutoStretchOnColumn := 1;
      Panels.Add(CurrentPanel);
    end;
    for j := 0 to ControlCount-1 do
    if (Controls[j] is TPanelCustomField) and (Controls[j] <> Controls[i]) then
    begin
      APanel := TPanelCustomField(Controls[j]);
      if APanel.AutoStretchHeight and not (
        (CurrentPanel.PosXRef >= APanel.PosXRef + APanel.WidthRef - 1) or // CurrentPanel after APanel
        (APanel.PosXRef >= CurrentPanel.PosXRef + CurrentPanel.WidthRef - 1) // CurrentPanel before APanel
        ) then
      begin
        Found := False;
        for k := 0 to Panels.Count-1 do
          with TPanelCustomField(Panels.Items[k]) do
            if not ((APanel.PosYRef >= PosYRef + HeightRef - 1) or
              (PosYRef >= APanel.PosYRef + APanel.HeightRef - 1)) then
              begin
                Found := True;
                Break;
              end;
        if not Found then
        begin
          Panels.Add(APanel);
          Inc(CurrentPanel.NbAutoStretchOnColumn);
          if (CurrentPanel.PosYRef >= APanel.PosYRef) then
            Inc(CurrentPanel.NbAutoStretchOnTop);
        end;
      end;
    end;

  end;
  Panels.Free;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TMovieFrameCustom.AddField(FieldProperties: TCustomFieldProperties): TPanelCustomField;
var
  PanelCF : TPanelCustomField;
begin
  Result := nil;

  if(FieldProperties = nil) then
    Exit;

  PanelCF := TPanelCustomField.Create(Self);
  PanelCF.Parent := Self;
  PanelCF.Init(FieldProperties);

  PanelCF.CFEnabled := (not ActionMoveResizeCustomFields.Checked) and FAllowEdit;
  PanelCF.CFBorder := ActionMoveResizeCustomFields.Checked;

  PanelCF.OnMouseDown := ControlMouseDown;
  PanelCF.OnMouseMove := ControlMouseMove;
  PanelCF.OnMouseUp := ControlMouseUp;
  PanelCF.OnFieldChange := FieldChange;
  PanelCF.OnUserAskListUpdate := UserAskListUpdate;
  PanelCF.OnURLButtonClick := FielURLButtonClick;
  PanelCF.OnURLEnter := FielURLEnter;
  PanelCF.FLabel.OnMouseUp := ControlMouseUp;

  Result := PanelCF;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetFieldsBorder(Value: Boolean);
var
  i: Integer;
begin
  for i := 0 to Self.ControlCount-1 do
  begin
    if (Self.Controls[i] is TPanelCustomField) then
      with TPanelCustomField(Self.Controls[i]) do
      begin
        CFBorder := ActionMoveResizeCustomFields.Checked;
        if (not ActionMoveResizeCustomFields.Checked) then
          Cursor := crDefault;
      end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetFieldsEnabled(Value: Boolean);
var
  i: Integer;
begin
  for i := 0 to Self.ControlCount-1 do
  begin
    if (Self.Controls[i] is TPanelCustomField) then
    begin
      TPanelCustomField(Self.Controls[i]).CFEnabled := Value;
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FrameResize(Sender: TObject);
var
  i: Integer;
  CurrentPanel: TPanelCustomField;
begin
  if(APanelCF = nil) and (FGeneratedFields) then
  begin
    for i := 0 to Self.ControlCount-1 do
      if (Self.Controls[i] is TPanelCustomField) then
      begin
        CurrentPanel := TPanelCustomField(Self.Controls[i]);
        if CurrentPanel.NbAutoStretchOnLeft > 0 then
          CurrentPanel.Left := Trunc(CurrentPanel.PosXRef + ((Self.Width - CF_FrameWidthRef) * CurrentPanel.NbAutoStretchOnLeft / CurrentPanel.NbAutoStretchOnLine))
        else
          CurrentPanel.Left := Trunc(CurrentPanel.PosXRef);
        if CurrentPanel.Left < 0 then CurrentPanel.Left := 0;

        if CurrentPanel.NbAutoStretchOnTop > 0 then
          CurrentPanel.Top := Trunc(CurrentPanel.PosYRef + ((Self.Height - CF_FrameHeightRef) * CurrentPanel.NbAutoStretchOnTop / CurrentPanel.NbAutoStretchOnColumn))
        else
          CurrentPanel.Top := Trunc(CurrentPanel.PosYRef);
        if CurrentPanel.Top < 0 then CurrentPanel.Top := 0;

        if CurrentPanel.AutoStretchWidth and (CurrentPanel.NbAutoStretchOnLine > 0) then
          CurrentPanel.Width := Trunc(CurrentPanel.WidthRef + ((Self.Width - CF_FrameWidthRef) / CurrentPanel.NbAutoStretchOnLine))
        else
          CurrentPanel.Width := Trunc(CurrentPanel.WidthRef);

        if CurrentPanel.AutoStretchHeight and (CurrentPanel.NbAutoStretchOnColumn > 0) then
          CurrentPanel.Height := Trunc(CurrentPanel.HeightRef + ((Self.Height - CF_FrameHeightRef) / CurrentPanel.NbAutoStretchOnColumn))
        else
          CurrentPanel.Height := Trunc(CurrentPanel.HeightRef);
      end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionAddCustomFieldExecute(Sender: TObject);
begin
  if Assigned(FOnFieldAdd) then
    FOnFieldAdd(nil);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionModifyCustomFieldExecute(Sender: TObject);
begin
  if Assigned(FOnFieldModify) then
    FOnFieldModify(FSelectedPanel);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionDeleteCustomFieldExecute(Sender: TObject);
begin
  if Assigned(FOnFieldDelete) then
    FOnFieldDelete(FSelectedPanel);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionMoveResizeCustomFieldsExecute(Sender: TObject);
begin
  ActionMoveResizeCustomFields.Checked := not ActionMoveResizeCustomFields.Checked;
  SetFieldsBorder(ActionMoveResizeCustomFields.Checked);
  SetFieldsEnabled((not ActionMoveResizeCustomFields.Checked) and FAllowEdit);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionDefaultPositioningExecute(Sender: TObject);
var
  Vis: Boolean;
  i: Integer;
begin
  VertScrollBar.Position := 0;
  HorzScrollBar.Position := 0;
  CF_FrameWidthRef := Width;
  CF_FrameHeightRef := Height;
  for i := 0 to ControlCount-1 do
    if Controls[i] is TPanelCustomField then
      with TPanelCustomField(Controls[i]) do
      begin
        PosYRef := -1;
        PosXRef := -1;
        WidthRef := CF_PanelWidth;
        if (FType <> ftText) then HeightRef := CF_PanelHeight
        else HeightRef := CF_PanelHeightText;
        AutoStretchWidth := True;
        AutoStretchHeight := False;
        CFLabelWidth := CF_LabelWidth;
      end;
  FGeneratedFields := False;
  Vis := Visible;
  Visible := False;
  CheckFieldsPosition;
  ComputeFieldsValue;
  Visible := Vis;
  FGeneratedFields := True;
  FrameResize(Self);
  FieldPropertiesChange(nil);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionAutoStretchWidthExecute(Sender: TObject);
begin
  FSelectedPanel.AutoStretchWidth := not FSelectedPanel.AutoStretchWidth;
  UpdateFieldsProperties;
  FieldPropertiesChange(FSelectedPanel);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ActionAutoStretchHeightExecute(Sender: TObject);
begin
  FSelectedPanel.AutoStretchHeight := not FSelectedPanel.AutoStretchHeight;
  UpdateFieldsProperties;
  FieldPropertiesChange(FSelectedPanel);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ControlMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  rec: TRect;
  pt: TPoint;
begin
  if (ActionMoveResizeCustomFields.Checked) and (Button = mbLeft) then
  begin
    if (Sender is TPanelCustomField) then
    begin
      APanelCF := TPanelCustomField(Sender);
      pt := ScreenToClient(Mouse.CursorPos);
      rec := APanelCF.BoundsRect;
      //if ((pt.X > rec.Right - 10) and (pt.Y > rec.Bottom - 10)) then
      if (pt.X > rec.Right - 8) then
      begin //resize panel right
        Resizing := 1;
      end
      else if ((pt.X > rec.Left + APanelCF.CFLabelWidth + CF_PanelMarginBetween - 3) and
        (pt.X < rec.Left + APanelCF.CFLabelWidth + CF_PanelMarginBetween + 3)) then
      begin //resize label
        Resizing := 2;
      end
      else if (pt.X < rec.Left + 7) then
      begin //resize label left
        Resizing := 3;
      end
      else
      begin //move label
        Resizing := 0;
      end;
      SetCapture(APanelCF.Handle);
      GetCursorPos(oldPos);
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ControlMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  newPos: TPoint;
  frmPoint : TPoint;
  rec : TRect;
begin
  if APanelCF <> nil then
  begin
    with APanelCF do
    begin
      GetCursorPos(newPos);

      if Resizing = 1 then
      begin //resize panel right
        if APanelCF.FType <> ftText then
        begin
          if Width - oldPos.X + newPos.X > CF_PanelMinWidth then
          begin
            Width := Width - oldPos.X + newPos.X;
            oldPos := newPos;
          end
        end
        else
        begin
          if (Width - oldPos.X + newPos.X > CF_PanelMinWidth) or
            (Height - oldPos.Y + newPos.Y > CF_PanelMinHeight) then
          begin
            if (Width - oldPos.X + newPos.X > CF_PanelMinWidth) then
              Width := Width - oldPos.X + newPos.X;
            if (Height - oldPos.Y + newPos.Y > CF_PanelMinHeight) then
              Height := Height - oldPos.Y + newPos.Y;
            oldPos := newPos;
          end
        end
      end
      else if Resizing = 2 then
      begin //resize label
        if CFLabelWidth - oldPos.X + newPos.X > CF_LabelMinWidth then
        begin
          CFLabelWidth := CFLabelWidth - oldPos.X + newPos.X;
          oldPos := newPos;
        end;
      end
      else if Resizing = 3 then
      begin //resize panel left
        if Width + oldPos.X - newPos.X > CF_PanelMinWidth then
        begin
          Width := Width + oldPos.X - newPos.X;
          Left := Left - oldPos.X + newPos.X;
          oldPos := newPos;
        end;
      end
      else
      begin //move panel
        Left := Left - oldPos.X + newPos.X;
        Top := Top - oldPos.Y + newPos.Y;
        oldPos := newPos;
      end;
    end;
  end
  else if (ActionMoveResizeCustomFields.Checked) then
  begin
    frmPoint := ScreenToClient(Mouse.CursorPos);
    if (Sender is TPanelCustomField) then
    begin
      APanelCF := TPanelCustomField(Sender);
      APanelCF.BringToFront;
      rec := APanelCF.BoundsRect;
      //if ((frmPoint.X > rec.Right - 10) and (frmPoint.Y > rec.Bottom - 10)) then
      if (frmPoint.X > rec.Right - 8) then
        if APanelCF.FType <> ftText then
          APanelCF.Cursor := crSizeWE //resize panel cursor right
        else
          APanelCF.Cursor := crSizeNWSE //resize panel cursor right/bottom
      else if ((frmPoint.X > rec.Left + APanelCF.CFLabelWidth + CF_PanelMarginBetween - 3) and
        (frmPoint.X < rec.Left + APanelCF.CFLabelWidth + CF_PanelMarginBetween+ + 3)) then
        APanelCF.Cursor := crSizeWE //resize label cursor
      else if (frmPoint.X < rec.Left + 7) then
          APanelCF.Cursor := crSizeWE //resize panel cursor left
      else
        APanelCF.Cursor := crSize; //move panel cursor
      APanelCF := nil;
    end;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.ControlMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  pt: TPoint;
begin
  if (APanelCF <> nil) and (Button = mbLeft) then
  begin
    with APanelCF do
    begin
      if Left < 0 then
        Left := 0;
      if Top < 0 then
        Top := 0;
    end;
    ReleaseCapture;
    UpdateFieldsProperties;
    FieldPropertiesChange(APanelCF);
    APanelCF := nil;
  end
  else if (ActionMoveResizeCustomFields.Checked) and (Button = mbRight) and (APanelCF = nil) then
  begin
    if (Sender is TPanelCustomField) then
    begin
      FSelectedPanel := TPanelCustomField(Sender);
      ActionAutoStretchWidth.Enabled := True;
      ActionAutoStretchWidth.Visible := ActionAutoStretchWidth.Enabled;
      ActionAutoStretchHeight.Enabled := FSelectedPanel.FType = ftText;
      ActionAutoStretchHeight.Visible := ActionAutoStretchHeight.Enabled;
      ActionAutoStretchWidth.Checked := FSelectedPanel.AutoStretchWidth;
      ActionAutoStretchHeight.Checked := FSelectedPanel.AutoStretchHeight;
      ActionDeleteCustomField.Enabled := True;
      ActionDeleteCustomField.Visible := ActionDeleteCustomField.Enabled;
      ActionModifyCustomField.Enabled := True;
      ActionModifyCustomField.Visible := ActionDeleteCustomField.Enabled;
      TBSep2.Visible := True;
      GetCursorPos(pt);
      PopupCustomField.Popup(pt.X, pt.Y);
    end;
  end
  else if (not ActionMoveResizeCustomFields.Checked) and (APanelCF = nil) then
    FrameMouseUp(Sender, Button, Shift, X, Y);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FrameMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  pt: TPoint;
  i: Integer;
begin
  if(Button = mbRight) then
  begin
    FSelectedPanel := nil;
    if (Sender is TLabel) and (TLabel(Sender).Parent is TPanelCustomField) then
      FSelectedPanel :=  TPanelCustomField(TLabel(Sender).Parent)
    else
    begin
      GetCursorPos(pt);
      pt := ScreenToClient(pt);
      for i := 0 to ControlCount-1 do
      begin
        if (Controls[i] is TPanelCustomField) and
          (pt.X > Controls[i].Left) and (pt.X < Controls[i].Left + Controls[i].Width) and
          (pt.Y > Controls[i].Top) and (pt.Y < Controls[i].Top + Controls[i].Height) then
        begin
          FSelectedPanel :=  TPanelCustomField(Controls[i]);
          break;
        end;
      end;
    end;
    ActionDeleteCustomField.Enabled := FSelectedPanel <> nil;
    ActionDeleteCustomField.Visible := ActionDeleteCustomField.Enabled;
    ActionModifyCustomField.Enabled := FSelectedPanel <> nil;
    ActionModifyCustomField.Visible := ActionDeleteCustomField.Enabled;
    ActionAutoStretchWidth.Enabled := False;
    ActionAutoStretchWidth.Visible := ActionAutoStretchWidth.Enabled;
    ActionAutoStretchHeight.Enabled := False;
    ActionAutoStretchHeight.Visible := ActionAutoStretchHeight.Enabled;
    TBSep2.Visible := False;
    GetCursorPos(pt);
    PopupCustomField.Popup(pt.X, pt.Y);
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}
{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.EndUpdate;
begin
  Dec(FUpdateCount);
  if FUpdateCount < 0 then
    FUpdateCount := 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TMovieFrameCustom.IsUpdating: Boolean;
begin
  Result := FUpdateCount > 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.Translate;
var
  i: Integer;
begin
  Translator.Translate(Self);
  for i := 0 to Self.ControlCount-1 do
    if (Self.Controls[i] is TPanelCustomField) then
      with TPanelCustomField(Self.Controls[i]) do
        CFName := CFName; // Regenerate label name (for ':' and ' :' according to language)
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetDefaultValues;
var
  i: Integer;
begin
  if Properties = nil then
    Exit;
  BeginUpdate;
  try
    with Properties do
    begin
      for i:= 0 to Count-1 do
      begin
        if Properties.Objects[i].FieldObject <> nil then
          with TPanelCustomField(Properties.Objects[i].FieldObject) do
          begin
            if (Self.Properties.Objects[i].FieldType = ftDate) and
              (AnsiSameText(Self.Properties.Objects[i].DefaultValue, 'Today')) then
              CFValue := DateToStr(Date, FormatSettings)
            else
              CFValue := Self.Properties.Objects[i].DefaultValue;
          end;
        //else
        //  showmessage('DEGUG: Regenerate fields after changes forgotten ! (1)');
      end;
    end;
  finally
    EndUpdate;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.LoadFromObject(AMovie: TMovie);
var
  i: Integer;
begin
  if Properties = nil then
    Exit;
  BeginUpdate;
  try
    with Properties do
    begin
      for i:= 0 to Count-1 do
      begin
        if Properties.Objects[i].FieldObject <> nil then
          with TPanelCustomField(Properties.Objects[i].FieldObject) do
          begin
            CFValue := AMovie.CustomFields.GetFieldValue(Strings[i]);
          end;
        //else
        //  showmessage('DEBUG: Regenerate fields after changes forgotten ! (2)');
      end;
    end;
  finally
    EndUpdate;
  end;
  FModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SaveToObject(AMovie: TMovie; const ForceSave: Boolean = False);
var
  i: Integer;
begin
  if Properties = nil then
    Exit;
  if Modified or ForceSave then
    with Properties do
    begin
      for i:= 0 to Count-1 do
      begin
        if Properties.Objects[i].FieldObject <> nil then
          with TPanelCustomField(Properties.Objects[i].FieldObject) do
            AMovie.CustomFields.SetFieldValue(Strings[i], CFValue);
        //else
        //  showmessage('DEBUG: Regenerate fields after changes forgotten ! (3)');
      end;
    end;
  if not ForceSave then
    FModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TMovieFrameCustom.GetCurrentValue(FieldTag: string): string;
var
  FieldProperties: TCustomFieldProperties;
begin
  if Properties = nil then
    Exit;
  FieldProperties := Properties.GetField(FieldTag);
  if (FieldProperties <> nil) and (FieldProperties.FieldObject <> nil) then
    Result := TPanelCustomField(FieldProperties.FieldObject).CFValue
  else
    Result := '';
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetCurrentValue(FieldTag: string; FieldValue: string);
var
  FieldProperties: TCustomFieldProperties;
begin
  if Properties = nil then
    Exit;
  FieldProperties := Properties.GetField(FieldTag);
  if (FieldProperties <> nil) and (FieldProperties.FieldObject <> nil) then
    TPanelCustomField(FieldProperties.FieldObject).CFValue := FieldValue;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TMovieFrameCustom.FindURLControl(str: string): TAntJvComboEditXP;
var
  i: Integer;
begin
  Result := nil;
  for i := 0 to ControlCount-1 do
    if Controls[i] is TPanelCustomField then
      with TPanelCustomField(Controls[i]) do
        if (FType = ftURL) and (FControl is TAntJvComboEditXP) and
          AnsiSameText(TAntJvComboEditXP(FControl).Text, str) then
          begin
            Result := TAntJvComboEditXP(FControl);
            break;
          end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.SetAllowEdit(Value: Boolean);
begin
  FAllowEdit := Value;
  SetFieldsEnabled((not ActionMoveResizeCustomFields.Checked) and FAllowEdit);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FieldChange(Sender: TObject);
begin
  if not IsUpdating then
  begin
    FModified := True;
    if Assigned(FOnFieldChange) then
      FOnFieldChange(Sender);
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FieldPropertiesChange(Sender: TObject);
begin
  FPropertiesModified := True;
  if Assigned(FOnFieldPropertiesChange) then
      FOnFieldPropertiesChange(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.UserAskListUpdate(Sender: TObject);
begin
  if Assigned(FOnUserAskListUpdate) then
      FOnUserAskListUpdate(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FielURLButtonClick(Sender: TObject);
begin
  if Assigned(OnURLButtonClick) then
    OnURLButtonClick(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrameCustom.FielURLEnter(Sender: TObject);
begin
  if Assigned(OnURLEnter) then
    OnURLEnter(Sender);
end;

end.
